<?php
require_once '../config.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Handle license actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $domain = sanitizeInput($_POST['domain']);
                $expires_at = !empty($_POST['expires_at']) ? $_POST['expires_at'] : null;
                $license_key = generateLicenseKey();
                
                $stmt = $pdo->prepare("INSERT INTO licenses (license_key, domain, expires_at) VALUES (?, ?, ?)");
                $stmt->execute([$license_key, $domain, $expires_at]);
                break;
                
            case 'delete':
                $license_id = (int)$_POST['license_id'];
                $stmt = $pdo->prepare("DELETE FROM licenses WHERE id = ?");
                $stmt->execute([$license_id]);
                break;
                
            case 'toggle_status':
                $license_id = (int)$_POST['license_id'];
                $new_status = $_POST['status'] === 'active' ? 'inactive' : 'active';
                $stmt = $pdo->prepare("UPDATE licenses SET status = ? WHERE id = ?");
                $stmt->execute([$new_status, $license_id]);
                break;
        }
    }
}

// Get all licenses
$stmt = $pdo->query("SELECT * FROM licenses ORDER BY created_at DESC");
$licenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>License Management - Admin Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.7.2/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container">
            <a class="navbar-brand" href="#">License System</a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="logout.php">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="row mb-4">
            <div class="col">
                <h2>License Management</h2>
            </div>
            <div class="col text-end">
                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createLicenseModal">
                    <i class="bi bi-plus-circle"></i> Create New License
                </button>
            </div>
        </div>

        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>License Key</th>
                                <th>Domain</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Expires</th>
                                <th>Last Check</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($licenses as $license): ?>
                            <tr>
                                <td><?php echo $license['license_key']; ?></td>
                                <td><?php echo $license['domain']; ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $license['status'] === 'active' ? 'success' : 'danger'; ?>">
                                        <?php echo ucfirst($license['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo date('Y-m-d H:i', strtotime($license['created_at'])); ?></td>
                                <td><?php echo $license['expires_at'] ? date('Y-m-d', strtotime($license['expires_at'])) : 'Never'; ?></td>
                                <td><?php echo $license['last_check'] ? date('Y-m-d H:i', strtotime($license['last_check'])) : 'Never'; ?></td>
                                <td>
                                    <form method="POST" class="d-inline">
                                        <input type="hidden" name="action" value="toggle_status">
                                        <input type="hidden" name="license_id" value="<?php echo $license['id']; ?>">
                                        <input type="hidden" name="status" value="<?php echo $license['status']; ?>">
                                        <button type="submit" class="btn btn-sm btn-<?php echo $license['status'] === 'active' ? 'warning' : 'success'; ?>">
                                            <?php echo $license['status'] === 'active' ? 'Deactivate' : 'Activate'; ?>
                                        </button>
                                    </form>
                                    <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this license?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="license_id" value="<?php echo $license['id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Create License Modal -->
    <div class="modal fade" id="createLicenseModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Create New License</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="create">
                        <div class="mb-3">
                            <label for="domain" class="form-label">Domain</label>
                            <input type="text" class="form-control" id="domain" name="domain" required>
                        </div>
                        <div class="mb-3">
                            <label for="expires_at" class="form-label">Expiration Date (Optional)</label>
                            <input type="date" class="form-control" id="expires_at" name="expires_at">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary">Create License</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 